unit Main;

interface

uses
  Winapi.Windows, Winapi.Messages, System.SysUtils, System.Variants, System.Classes, Vcl.Graphics,
  Vcl.Controls, Vcl.Forms, Vcl.Dialogs, Vcl.StdCtrls, Vcl.ComCtrls, JPEG, JpegDec, SynGdiPlus, {NativeJpg, sdJpegTypes,}
  System.Generics.Collections, uSMBios, ClipBrd;

Const
      crlf = #13#10;

Type

     // Normally would use a TStream for maximum flexibility, but using a TMemoryStream instead
     // as an optimization to save copying memory contents unnecessarily!

     TJpegDecoder = Class(TObject)
       Class Function DecoderName: String; Virtual; Abstract;
       Function Decode(Stream: TMemoryStream): TBitmap; Virtual; Abstract;
     end;

     TJpegDecoderClass = Class of TJpegDecoder;

     TJpegDecoderClasses = TList<TJpegDecoderClass>;

     TJpegDecDecoder = Class(TJpegDecoder)
       Class Function DecoderName: String; Override;
       Function Decode(Stream: TMemoryStream): TBitmap; Override;
     end;

     TDelphiJpegDecoder = Class(TJpegDecoder)
       Class Function DecoderName: String; Override;
       Function Decode(Stream: TMemoryStream): TBitmap; Override;
     end;

     TSynGdiPlusJpegDecoder = Class(TJpegDecoder)
       Class Function DecoderName: String; Override;
       Function Decode(Stream: TMemoryStream): TBitmap; Override;
     end;

     // I had included the NativeJPG library, but despite supposedly being
     // almost as fast as JpegDec, it seems to be slower than the
     // standard Delphi Library!
{
     TNativeJpgJpegDecoder = Class(TJpegDecoder)
       Class Function DecoderName: String; Override;
       Function Decode(Stream: TMemoryStream): TBitmap; Override;
     end;
}
     TTestResult = Class(TObject)
       DecoderName: String;
       SingleThreadedResult: Integer;
       MultiThreadedResult: Integer;

       Procedure ToListItem(ListItem: TListItem);
       Function ToString: String; Override;
     end;

     TTestResults = TObjectList<TTestResult>;


type
  TMainForm = class(TForm)
    Label1: TLabel;
    SourceFilename: TEdit;
    BrowseSource: TButton;
    Label2: TLabel;
    IterationCount: TEdit;
    Label3: TLabel;
    ThreadCount: TEdit;
    TestResultsListView: TListView;
    Run: TButton;
    Close: TButton;
    StatusBar1: TStatusBar;
    procedure BrowseSourceClick(Sender: TObject);
    procedure CloseClick(Sender: TObject);
    procedure RunClick(Sender: TObject);
    procedure FormCreate(Sender: TObject);
  private
    { Private declarations }
  public
    { Public declarations }
  end;

var
  MainForm: TMainForm;

implementation

{$R *.dfm}

var  DecoderClasses: TJpegDecoderClasses;

var RemainingCount: Integer;

Procedure DoDecoding(JpegDecoder: TJpegDecoder; MemoryStream: TBytesStream; Iterations: Integer);
var Bitmap: TBitmap;
    Stream: TBytesStream;
begin
  Stream:= TBytesStream.Create(MemoryStream.Bytes);

  MemoryStream.Seek(0, soFromBeginning);

  Stream.LoadFromStream(MemoryStream);

  TThread.CreateAnonymousThread(
    Procedure
    var i: Integer;
    begin
      try
        for i:= 1 to Iterations do
          begin
            Bitmap:= JpegDecoder.Decode(Stream);
            Bitmap.Free;

            if Application.Terminated then
              Break;
          end;

        InterlockedDecrement(RemainingCount);
      finally
        Stream.Free;
      end;
    end).Start;
end;

Function TimeToDecode(JpegDecoder: TJpegDecoder; MemoryStream: TBytesStream; Iterations: Integer; ThreadCount: Integer): Integer;
var Start: DWord;
    i: Integer;
begin
  RemainingCount:= ThreadCount;

  Start:= GetTickCount;

  for i:= 1 to ThreadCount do
    DoDecoding(JpegDecoder, MemoryStream, Iterations div ThreadCount );

  while RemainingCount > 0 do
    begin
      Application.ProcessMessages;
      Sleep(10);
    end;

  Result:= GetTickCount - Start;
end;

Function RunTest(DecoderClass: TJpegDecoderClass; FileName: String; Iterations: Integer; ThreadCount: Integer): TTestResult;
var MemoryStream: TBytesStream;
    JpegDecoder: TJpegDecoder;
    Duration: DWord;
begin
  MemoryStream:= TBytesStream.Create;
  JpegDecoder:= DecoderClass.Create;

  try
    MemoryStream.LoadFromFile(FileName);

    Result:= TTestResult.Create;
    Result.DecoderName:= JpegDecoder.DecoderName;

    Duration:= TimeToDecode(JpegDecoder, MemoryStream, Iterations, 1);
    Result.SingleThreadedResult:= Round(Iterations * 1000 / Duration);

    Duration:= TimeToDecode(JpegDecoder, MemoryStream, Iterations, ThreadCount);
    Result.MultiThreadedResult:= Round(Iterations * 1000 / Duration);


  finally
    JpegDecoder.Free;
    MemoryStream.Free;
  end;
end;

procedure TMainForm.BrowseSourceClick(Sender: TObject);
var OpenDialog: TOpenDialog;
begin
  OpenDialog:= TOpenDialog.Create(self);

  try
    OpenDialog.DefaultExt:= 'jpg';
    OpenDialog.Filter:= 'JPEG Files (*.jpg)|*.JPG';
    OpenDialog.Options:= [ofFileMustExist];

    if OpenDialog.Execute then
      SourceFilename.Text:= OpenDialog.FileName;
  finally
    OpenDialog.Free;
  end;
end;

procedure TMainForm.CloseClick(Sender: TObject);
begin
  Application.Terminate;
end;

procedure TMainForm.FormCreate(Sender: TObject);
begin
  ThreadCount.Text:= IntToStr(CPUCount);
end;

procedure TMainForm.RunClick(Sender: TObject);
var TestResult: TTestResult;
    ListItem: TListItem;
    Iterations: Integer;
    Threads: Integer;
    DecoderClass: TJpegDecoderClass;
    Results: String;
    SMBios: TSMBios;
    ProcessorInfo: TProcessorInformation;
begin
  TestResultsListView.Clear;

  Iterations:= StrToIntDef(IterationCount.Text, 1000);
  Threads:= StrToIntDef(ThreadCount.Text, 8);

  for DecoderClass in DecoderClasses do
    begin
      StatusBar1.SimpleText:= 'Running test for: ' + DecoderClass.DecoderName;
      TestResult:= RunTest(DecoderClass, SourceFileName.Text, Iterations, Threads);

      try
        if Application.Terminated then
          exit;

        ListItem:= TestResultsListView.Items.Add;
        TestResult.ToListItem(ListItem);
        Results:= Results + crlf + TestResult.ToString;
      finally
        TestResult.Free;
      end;
    end;

  StatusBar1.SimpleText:= 'Finished!';

  if MessageBox(Application.Handle, 'Finished!'+crlf+crlf+'Do you want to copy the results to the clipboard?', 'Finished', MB_YesNo) = IDYes then
    begin
      SMBios:= TSMBios.Create;

      try
        for ProcessorInfo in SMBios.ProcessorInfo do
          begin
            Results:= 'CPU: '+String(ProcessorInfo.ProcessorVersionStr) + crlf +
                      'Core Count: '+IntToStr(ProcessorInfo.RAWProcessorInformation.CoreCount) + crlf +
                      'Thread Count: '+IntToStr(ProcessorInfo.RAWProcessorInformation.ThreadCount) + crlf +
                      Results;
          end;
      finally
        SMBios.Free;
      end;

      Results:= 'Iterations: ' + IntToStr(Iterations) + crlf +
                'Thread Count: ' + IntToStr(Threads) + crlf + crlf +
                Results;

      ClipBoard.AsText:= Results;
    end;
end;

{ TJpegDecDecoder }

function TJpegDecDecoder.Decode(Stream: TMemoryStream): TBitmap;
begin
  try
    Result:= JpegDecode(Stream.Memory, Stream.Size);
  except on e: Exception do
    Result:= nil;
  end;
end;

Class Function TJpegDecDecoder.DecoderName: String;
begin
  Result:= 'JpegDec';
end;

{ TTestResult }

procedure TTestResult.ToListItem(ListItem: TListItem);
begin
  ListItem.Caption:= DecoderName;

  ListItem.SubItems.Clear;
  ListItem.SubItems.Add(IntToStr(SingleThreadedResult));
  ListItem.SubItems.Add(IntToStr(MultiThreadedResult));
  ListItem.SubItems.Add(IntToStr(Round((MultiThreadedResult/SingleThreadedResult)*100))+'%');
end;

function TTestResult.ToString: String;
begin
  Result:= DecoderName+': '+crlf+
           'Single Threaded Performance: '+IntToStr(SingleThreadedResult)+' images per second'+crlf+
           'Multithreaded Performance:   '+IntToStr(MultiThreadedResult)+' images per second'+crlf+
           'Relative Performance: '+IntToStr(Round((MultiThreadedResult/SingleThreadedResult)*100))+'%'+crlf;
end;

{ TDelphiJpegDecoder }

function TDelphiJpegDecoder.Decode(Stream: TMemoryStream): TBitmap;
var JpegImage: Jpeg.TJpegImage;
begin
  JpegImage:= Jpeg.TJpegImage.Create;
  try
    Stream.Seek(0, soFromBeginning);
    JpegImage.LoadFromStream(Stream);
    Result:= TBitmap.Create;
    Result.Assign(JpegImage);
  finally
    JpegImage.Free;
  end;
end;

class function TDelphiJpegDecoder.DecoderName: String;
begin
  Result:= 'Standard Delphi';
end;

{ TSynGdiPlusJpegDecoder }

function TSynGdiPlusJpegDecoder.Decode(Stream: TMemoryStream): TBitmap;
var JpegImage: SynGdiPlus.TJpegImage;
begin
  JpegImage:= SynGdiPlus.TJpegImage.Create;
  try
    Stream.Seek(0, soFromBeginning);
    JpegImage.LoadFromStream(Stream);
    Result:= JpegImage.ToBitmap
  finally
    JpegImage.Free;
  end;
end;

class function TSynGdiPlusJpegDecoder.DecoderName: String;
begin
  Result:= 'SynGdiPlus';
end;

{ TNativeJpgJpegDecoder }
{
function TNativeJpgJpegDecoder.Decode(Stream: TMemoryStream): TBitmap;
var JpegImage: TsdJpegGraphic;
begin
  JpegImage:= TsdJpegGraphic.Create;
  try
    JpegImage.Performance:= jpBestSpeed;
    JPegImage.Scale:= jsDiv8;
    Stream.Seek(0, soFromBeginning);
    JpegImage.LoadFromStream(Stream);
    Result:= TBitmap.Create;
    Result.Assign(JpegImage);
  finally
    JpegImage.Free;
  end;
end;

class function TNativeJpgJpegDecoder.DecoderName: String;
begin
  Result:= 'NativeJPG';
end;
}

initialization
  DecoderClasses:= TJpegDecoderClasses.Create;

//  DecoderClasses.Add(TNativeJpgJpegDecoder);
  DecoderClasses.Add(TJpegDecDecoder);
  DecoderClasses.Add(TSynGdiPlusJpegDecoder);
  DecoderClasses.Add(TDelphiJpegDecoder);
finalization
  DecoderClasses.Free;
end.
